# Deployment Notes - Theme System & Localization Updates

## Overview
This deployment adds a complete theme system and fixes promotion item localization.

## Database Migration Required

### Step 1: Create Themes Table
Run this SQL on your production database:

```sql
CREATE TABLE IF NOT EXISTS themes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  background_color VARCHAR(50) NOT NULL,
  text_color VARCHAR(50) NOT NULL,
  accent_color VARCHAR(50) NOT NULL,
  cta_color VARCHAR(50) NOT NULL,
  font_style VARCHAR(100) NOT NULL,
  is_default BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  INDEX idx_themes_name (name)
);
```

### Step 2: Add Theme Column to Restaurants
```sql
ALTER TABLE restaurants 
ADD COLUMN selected_theme INT NULL,
ADD CONSTRAINT fk_restaurants_theme FOREIGN KEY (selected_theme) REFERENCES themes(id) ON DELETE SET NULL;
```

### Step 3: Seed Default Themes
```sql
INSERT INTO themes (id, name, background_color, text_color, accent_color, cta_color, font_style, is_default) VALUES 
(1, 'Default', '#ffffff', '#1f2937', '#f97316', '#f97316', 'system-ui, -apple-system, sans-serif', true),
(2, 'Classic & Elegant', '#faf9f6', '#1e293b', '#d4af37', '#8b0000', 'Georgia, serif', false),
(3, 'Casual & Friendly', '#f5f5dc', '#3e2723', '#ff6347', '#228b22', 'Verdana, sans-serif', false),
(4, 'Modern & Minimal', '#ffffff', '#000000', '#00bfa5', '#0091ea', 'Helvetica, Arial, sans-serif', false),
(5, 'Bold & Vibrant', '#fffaf0', '#1e3a8a', '#dc2626', '#16a34a', 'Montserrat, sans-serif', false),
(6, 'Dark Mode', '#1f2937', '#f3f4f6', '#3b82f6', '#ef4444', 'Inter, sans-serif', false)
ON DUPLICATE KEY UPDATE 
  background_color = VALUES(background_color),
  text_color = VALUES(text_color),
  accent_color = VALUES(accent_color),
  cta_color = VALUES(cta_color),
  font_style = VALUES(font_style),
  is_default = VALUES(is_default);
```

## Deployment Steps

1. **Backup Database**
   ```bash
   mysqldump -u user -p database_name > backup_$(date +%Y%m%d_%H%M%S).sql
   ```

2. **Pull Latest Code**
   ```bash
   cd /path/to/webapp
   git pull origin main
   ```

3. **Install Dependencies** (if any new packages)
   ```bash
   npm install
   ```

4. **Run Database Migrations**
   - Execute the SQL scripts above in order
   - Verify tables created: `SHOW TABLES LIKE 'themes';`
   - Verify column added: `DESCRIBE restaurants;`

5. **Test on Staging** (if available)
   - Access admin dashboard
   - Try creating/editing restaurant with theme selection
   - Visit live menu and verify theme loads

6. **Deploy to Production**
   ```bash
   pm2 restart webapp
   # or
   npm start
   ```

7. **Verify Deployment**
   - Check server logs: `pm2 logs webapp --nostream`
   - Test theme API: `curl http://localhost:3000/api/themes`
   - Test restaurant theme: `curl http://localhost:3000/api/restaurants/slug/YOUR_SLUG/theme`

## Rollback Plan

If issues occur:

1. **Remove Theme Column**
   ```sql
   ALTER TABLE restaurants DROP FOREIGN KEY fk_restaurants_theme;
   ALTER TABLE restaurants DROP COLUMN selected_theme;
   ```

2. **Drop Themes Table**
   ```sql
   DROP TABLE IF EXISTS themes;
   ```

3. **Revert Code**
   ```bash
   git revert HEAD
   pm2 restart webapp
   ```

## Testing Checklist

### Admin Dashboard
- [ ] Login as super admin
- [ ] Create new restaurant with theme selection
- [ ] Edit existing restaurant and change theme
- [ ] Login as restaurant admin
- [ ] Edit own restaurant and select theme
- [ ] Verify theme selector shows all 6 themes

### Live Menu
- [ ] Visit menu with default theme
- [ ] Visit menu with Classic & Elegant theme
- [ ] Visit menu with Casual & Friendly theme
- [ ] Visit menu with Modern & Minimal theme
- [ ] Visit menu with Bold & Vibrant theme
- [ ] Visit menu with Dark Mode theme
- [ ] Verify colors change correctly
- [ ] Check promotions display correctly
- [ ] Switch language (MK/EN) and verify promotion items

### API Endpoints
- [ ] GET /api/themes returns all themes
- [ ] GET /api/themes/1 returns default theme
- [ ] GET /api/restaurants/slug/:slug/theme returns restaurant theme
- [ ] POST /api/admin/restaurants with selected_theme saves correctly
- [ ] PUT /api/admin/restaurants/:id with selected_theme updates correctly

## Known Issues / Limitations

1. **Theme Preview**: No live preview before saving (future enhancement)
2. **Custom Themes**: Cannot create custom themes yet (use pre-defined only)
3. **Per-Category Colors**: Not yet implemented
4. **Font Loading**: Custom fonts may take time to load on first visit

## Performance Notes

- Theme loading adds 1 API call per menu page load
- CSS injection is minimal and happens once on page load
- No performance impact on existing functionality
- Cached in browser after first load

## Monitoring

Watch for:
- Database connection errors related to themes table
- API errors on `/api/themes` endpoints
- Console errors about theme application
- CSS not loading correctly on live menu

Check logs:
```bash
pm2 logs webapp --lines 100
```

## Support Contacts

For deployment issues:
- Database: [Your DBA contact]
- Backend: [Backend team]
- Frontend: [Frontend team]

## Deployment Date

**Date**: _____________
**Deployed By**: _____________
**Verified By**: _____________

## Post-Deployment Notes

_Add any notes about the deployment here after completion_
