
class RestaurantApp {
    constructor() {
        this.restaurants = [];
        this.init();
    }

    async init() {
        await this.loadRestaurants();
        this.render();
    }

    async loadRestaurants() {
        try {
            const response = await axios.get('/api/restaurants');
            if (response.data.success) {
                this.restaurants = response.data.data;
            } else {
                this.showError('Failed to load restaurants');
            }
        } catch (error) {
            console.error('Error loading restaurants:', error);
            this.showError('Failed to load restaurants');
        }
    }

    showError(message) {
        const app = document.getElementById('app');
        app.innerHTML = `
            <div class="max-w-4xl mx-auto p-6">
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                    <i class="fas fa-exclamation-triangle mr-2"></i>
                    ${message}
                </div>
            </div>
        `;
    }

    render() {
        const app = document.getElementById('app');
        
        if (this.restaurants.length === 0) {
            app.innerHTML = `
                <div class="max-w-4xl mx-auto p-6">
                    <div class="text-center py-12">
                        <i class="fas fa-utensils text-6xl text-gray-300 mb-4"></i>
                        <h2 class="text-2xl font-bold text-gray-600 mb-2">No Restaurants Available</h2>
                        <p class="text-gray-500">Please check back later.</p>
                    </div>
                </div>
            `;
            return;
        }

        app.innerHTML = `
            <!-- Header -->
            <div class="bg-white shadow-sm">
                <div class="max-w-7xl mx-auto px-4 py-6">
                    <div class="text-center">
                        <h1 class="text-4xl font-bold text-gray-900 mb-2">
                            <i class="fas fa-utensils mr-3 text-brand-orange"></i>
                            Restaurant Menu System
                        </h1>
                        <p class="text-xl text-gray-600">Choose a restaurant to view their menu</p>
                    </div>
                </div>
            </div>

            <!-- Restaurant Grid -->
            <div class="max-w-7xl mx-auto px-4 py-8">
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    ${this.restaurants.map(restaurant => this.renderRestaurantCard(restaurant)).join('')}
                </div>
            </div>

            <!-- Admin Link -->
            <div class="fixed bottom-6 right-6">
                <a href="/admin" class="bg-gray-800 text-white px-4 py-2 rounded-lg shadow-lg hover:bg-gray-700 transition-colors">
                    <i class="fas fa-cog mr-2"></i>Admin
                </a>
            </div>
        `;
    }

    renderRestaurantCard(restaurant) {
        const defaultImage = 'https://images.unsplash.com/photo-1555396273-367ea4eb4db5?w=400&h=300&fit=crop';
        
        return `
            <div class="bg-white rounded-lg shadow-md hover:shadow-lg transition-shadow overflow-hidden cursor-pointer" 
                 onclick="window.location.href='/menu/${restaurant.id}'">
                <div class="h-48 bg-gray-200 overflow-hidden">
                    <img src="${restaurant.logo_url || defaultImage}" 
                         alt="${restaurant.name}"
                         class="w-full h-full object-cover"
                         onerror="this.src='${defaultImage}'">
                </div>
                
                <div class="p-6">
                    <h3 class="text-xl font-bold text-gray-900 mb-2">${restaurant.name}</h3>
                    
                    ${restaurant.description ? `
                        <p class="text-gray-600 mb-3 line-clamp-2">${restaurant.description}</p>
                    ` : ''}
                    
                    <div class="space-y-2 text-sm text-gray-500">
                        ${restaurant.address ? `
                            <div class="flex items-center">
                                <i class="fas fa-map-marker-alt mr-2 w-4"></i>
                                <span>${restaurant.address}</span>
                            </div>
                        ` : ''}
                        
                        ${restaurant.phone ? `
                            <div class="flex items-center">
                                <i class="fas fa-phone mr-2 w-4"></i>
                                <span>${restaurant.phone}</span>
                            </div>
                        ` : ''}
                        
                        ${restaurant.email ? `
                            <div class="flex items-center">
                                <i class="fas fa-envelope mr-2 w-4"></i>
                                <span>${restaurant.email}</span>
                            </div>
                        ` : ''}
                    </div>
                    
                    <div class="mt-4 pt-4 border-t border-gray-200">
                        <div class="flex items-center justify-between">
                            <span class="text-sm text-gray-500">View Menu</span>
                            <i class="fas fa-arrow-right text-brand-orange"></i>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
}
document.addEventListener('DOMContentLoaded', () => {
    new RestaurantApp();
});